#=====================================================
#     Self contained utility functions
#=====================================================

# This function deals with incompatibilities between how 
# the argument file is written as a text file with .R extension 
# and what R expects. For 
# example it casts "NULL" as NULL etc.
# test: "list(mRNA="Dynamic",protein="Dynamic")"


source.arguments <- function(){
  
  #Re-write lists in correct R format
  #-----------------------------------
  txt.arg.input <- readLines("./arguments.r")
  if(!file.exists("arguments_raw.r"))
    writeLines(text = noquote(txt.arg.input), con = "arguments_raw.r")
  list.indices <- grep(pattern ="list([(]{1})",x = txt.arg.input)
  for(i in list.indices){
    var.name <- unlist(strsplit(txt.arg.input[i],split = "\\\""))[1]
    if(length(grep(pattern ="list([(]{1})",x = var.name))!=0)
      next
    
    var.val <- paste(unlist(strsplit(txt.arg.input[i],split = "="))[-1],collapse = "=")
    
    n.components <- sum(unlist(strsplit(var.val,split = ""))=="=")
    component.vals <- lapply(unlist(strsplit(var.val,"=|,|)"))[(1:n.components)*2],function(x)x)
    component.vals[grep("[0-9]",unlist(component.vals))] <- as.numeric(component.vals[grep("[0-9]",unlist(component.vals))])
    component.names <- unlist(strsplit(var.val,"=|\\(|,"))[(1:n.components)*2]
    txt.arg.input[i] <-paste(var.name,"list(",paste(paste(component.names,component.vals,sep="="),collapse = ","),")",sep="")
  }
  
  #Replace "NUL" with NULL 
  txt.arg.input <- gsub(pattern = "(=\"NULL\")|(= \"NULL\")",replacement = noquote("= NULL"),x = txt.arg.input)
  writeLines(text = noquote(txt.arg.input), con = "arguments.r")
  
  

  tmp.env <- new.env()
  source("./arguments.r",local = tmp.env)

  
  # Add the hardcoded variables:
  #-----------------------------
  if(!exists("probe.annotation.KEGG.id.column",envir = tmp.env))
    assign("probe.annotation.KEGG.id.column" ,value = "KEGG.Pathways",envir = tmp.env)
  if(!exists("probe.annotation.KEGG.Gene.Name",envir = tmp.env))
    assign("probe.annotation.KEGG.Gene.Name" ,value = "Official.Gene.Name",envir = tmp.env)
  if(!exists("min.KEGG.set.size",envir = tmp.env))
    assign("min.KEGG.set.size" ,value = 1,envir = tmp.env)
  if(!exists("celltypes.arg.filePath",envir = tmp.env))
    assign("celltypes.arg.filePath" ,value = "./cell type contrasts matrix.csv",envir = tmp.env)
  
  
  # Make new convenience variables based on argument
  #-------------------------------------------------
  assign("normmodule.arg.run" ,value = !is.null(tmp.env$normmodule.analytes.run),envir = tmp.env)
  
  
  # Change varnames
  #----------------
  if(exists("probeAnnotation",envir = tmp.env)){
    assign("path.probe.annot" ,value = get("probeAnnotation",envir = tmp.env),envir = tmp.env)
    rm(list = c("path.geneannot","probeAnnotation"),envir = tmp.env)
  }
  if(exists("gene.annotation.id.column",envir = tmp.env)){
    assign("probe.annotation.id.column" ,value = "ProbeID",envir = tmp.env)
    rm(list = "gene.annotation.id.column",envir = tmp.env)
  }else{# TEMPORARY HARDCOING OF PROBE ID
    assign("probe.annotation.id.column" ,value = "ProbeID",envir = tmp.env)
  }
  
  if(exists("min.gene.covariate",envir =  tmp.env)){
    assign("min.prb.set.size" ,value = get("min.gene.covariate",envir = tmp.env),envir = tmp.env)
    rm(list = "min.gene.covariate",envir = tmp.env)
  }else{# TEMPORARY HARDCODING OF MIN NUMBER OF PROBES IN A SET. DEFAULT IS NEEDED FROM UI
    assign("min.prb.set.size" ,value = 5, envir = tmp.env)
  }
  
  if(exists("deregmodule.arg.run",envir =  tmp.env)){
    assign("pathscores.arg.run" ,value = get("deregmodule.arg.run",envir = tmp.env),envir = tmp.env)
    rm(list = "deregmodule.arg.run",envir = tmp.env)
  }
  
  if(exists("deregmodule.arg.method",envir =  tmp.env)){
    assign("pathscores.arg.method" ,value = get("deregmodule.arg.method",envir = tmp.env),envir = tmp.env)
    rm(list = "deregmodule.arg.method",envir = tmp.env)
  }
  
  if(exists("covariates.for.pathscoring",envir =  tmp.env)){
    assign("pathscores.arg.covariates" ,value = get("covariates.for.pathscoring",envir = tmp.env),envir = tmp.env)
    rm(list = "covariates.for.pathscoring",envir = tmp.env)
  }
  
  
  if(exists("deregmodule.arg.adjust.for",envir =  tmp.env)){
    assign("pathscores.arg.adjust.for" ,value = get("deregmodule.arg.adjust.for",envir = tmp.env),envir = tmp.env)
    rm(list = "deregmodule.arg.adjust.for",envir = tmp.env)
  }
  
  if(exists("celltypes.arg.geneselection",envir =  tmp.env)){
    assign("celltypes.arg.probeselection" ,value = get("celltypes.arg.geneselection",envir = tmp.env),envir = tmp.env)
    rm(list = "celltypes.arg.geneselection",envir = tmp.env)
  }
  
  
  if(exists("celltypes.arg.geneselection",envir =  tmp.env)){
    assign("celltypes.arg.probeselection" ,value = get("celltypes.arg.geneselection",envir = tmp.env),envir = tmp.env)
    rm(list = "celltypes.arg.geneselection",envir = tmp.env)
  }
  
  if(exists("celltypes.arg.filePath",envir =  tmp.env)){
    if(file.exists(get("celltypes.arg.filePath",envir = tmp.env))){
      assign("celltypes.arg.contrastsfile" ,value = get("celltypes.arg.filePath",envir = tmp.env),envir = tmp.env)
      rm(list = "celltypes.arg.filePath",envir = tmp.env)
    }
  }
  
  # Convert .png, .pdf etc. to png, pdf etc
  #----------------------------------------
  if(exists("plottypearg",envir =  tmp.env)){
    if(!is.null(tmp.env$plottypearg))
      tmp.env$plottypearg <- gsub(pattern = "\\.",replacement = "",tmp.env$plottypearg)
  }
  
  
  #======================================================
  #           TEMPORARY TURNING OFF of MODULES
  #======================================================
  #tmp.env$pathscores.arg.run<- FALSE
  #tmp.env$pathviewmodule.arg.run <- FALSE
  
  
  # Load tmp.env to global.env and cast NULL
  # Note: the casting of NULL may be redundant
  #       as we gsub above
  #-----------------------------------------
  arg.names <- ls(envir = tmp.env)
  
  for(arg.i in arg.names){
    
    tmp <- get(x = arg.i,envir = tmp.env)
    assign(x = arg.i,value = tmp,envir = globalenv())
    if(length(tmp)==1){
      
      if(is.character(tmp)){
        if(tmp == "NULL"){
          assign(x = arg.i,value = NULL,envir = globalenv())
          assign(x = arg.i,value = NULL,envir = tmp.env)
        }
          
        if(tmp == "NA"){
          assign(x = arg.i,value = NA,envir = globalenv())
          assign(x = arg.i,value = NA,envir = tmp.env)
        }
          
      } 
    } 
  }
  arg.env = tmp.env
  rm(tmp.env)
  return(arg.env)
  
}



run.analysis <- function(){
  
  warnings.paragraph <- ""
  
  #======================================================
  #           Load/Install required libraries
  #======================================================
  
  resultlib <- suppressWarnings(try(runloadlibraries(), silent = TRUE))
  warnings()
  if(class(resultlib) == "try-error")
  {
    cat("LOG:could not load libraries",file=log,sep='\n',append=TRUE)
    cat(resultlib,file=log,sep='\n',append=TRUE)
    cat("document.write('<p>Could not load libraries</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
    stop("could not load libraries")
  } else{
    print("Finished loading R libraries")
    cat("LOG:Finished loading R libraries",file=log,sep='\n\n',append=TRUE)
    cat("document.write('<p>Finished loading R libraries</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE) 
  }
  
  
  #======================================================
  #               Load and verify data
  #======================================================
  
  # FUTURE DESIGN REV: consider extracting "matrix set" information outside this step and at the time of use within the modules
  #                    for example, probe.set.matrix and KEGG.set.matrix etc. could be gleaned probe annotation by calling the 
  #                    proper functionality within load_and_clean.R
  
  annot.obj <- NULL
  annot.obj <- tryCatch(expr = construct.data.and.annotations(path.data = path.data,
                                                              path.sampleannot = path.sampleannot,
                                                              path.probeannot = path.probe.annot,
                                                              prb.annotation.id.column = probe.annotation.id.column,
                                                              
                                                              #prb.annotation.id.column = gene.annotation.id.column,
                                                              prb.annotation.KEGG.id.column = probe.annotation.KEGG.id.column,
                                                              prb.annotation.KEGG.Gene.Name = probe.annotation.KEGG.Gene.Name,
                                                              prb.annotation.celltype.id.column = probe.annotation.celltype.id.column,
                                                              prb.set.column = probe.set.column, # <------------------------------ comment this out to see the behavior when no probeset info is avaialable
                                                              min.prb.set.size = min.prb.set.size,
                                                              
                                                              #min.prb.set.size = min.gene.covariate,
                                                              min.KEGG.set.size = min.KEGG.set.size,
                                                              sample.annotation.id.column = sample.annotation.id.column,
                                                              sampleannot.variables = sampleannot.variables,
                                                              sampleannot.variabletypes = sampleannot.variabletypes,
                                                              sampleannot.referencelevels = sampleannot.referencelevels,
                                                              exclude.probes = NULL,
                                                              exclude.samples = NULL,
                                                              draw.color.legend = T,
                                                              min.samples = 2,
                                                              plottypearg = plottypearg,
                                                              log = log)
                        ,error = function(er){
                          ertmp <- paste("Data load and preprocess is aborted: the following error(s) occurred\n",er)
                          #cat(ertmp)
                          cat("document.write('<p> Data load and preprocess is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                          class(ertmp) <- "error"
                          return(ertmp)
                        }
                        ,finally = {})
  
  
  if(class(annot.obj) == "error" )
  {
    print("Failed in upload and preprocess")
    cat("LOG: Failed in upload and preprocess",file=log,sep='\n',append=TRUE)
    cat(annot.obj,file=log,sep='\n',append=TRUE)
    
    warnings.paragraph <- paste(warnings.paragraph,annot.obj,"\n")
    stop(warnings.paragraph)
  }else{
    # Assign to local variables
    #--------------------------
    rawdata <- annot.obj$rawdata
    processedData <- annot.obj$processed
    annot <- annot.obj$annot
    pannot <- annot.obj$pannot
    annotcols <- annot.obj$annotcols
    annotcols2 <- annot.obj$annotcols2
    sampleannot.variables <- annot.obj$sampleannot.variables
    sampleannot.variabletypes <- annot.obj$sampleannot.variabletypes
    sampleannot.referencelevels <- annot.obj$sampleannot.referencelevels
    prb.set.matrix <- annot.obj$prb.set.matrix
    celltype.set.matrix <- annot.obj$celltype.set.matrix
    warnings.paragraph <- annot.obj$warnings.paragraph
    KEGG.set.matrix <- annot.obj$KEGG.set.matrix
    prb.set.matrix.QC <- prb.set.matrix  # <-----------------------------------what's difference between these?
    
    #post hoc elimination of irrelvant warnings: 
    #This (along with other warning messaging) 
    #is to be handled by better design in next rev!
    #----------------------------------------------
    
    if(!celltypes.arg.run)
      warnings.paragraph <- gsub("Warning: No celltype probe annotation is provided\n","",warnings.paragraph)
    
    if(!pathviewmodule.arg.run)
      warnings.paragraph <- gsub("Warning: No KEGG id annotation is provided\n","",warnings.paragraph)
    
  }
  

  
  #======================================================
  #                   Normalization                
  #======================================================
  
  # This section could use some clean up and transferring the logic into DE source
  rawData.list  <- processedData.list <- list()
  length(rawData.list) <- length(processedData.list) <- nlevels(pannot$Analyte.Type)
  names(rawData.list) <- names(processedData.list) <- levels(pannot$Analyte.Type)
  
  
  normData.list <- rawData.list
  normData.list.2 <- list()  
  
  cat("running normalization module",file=log,sep='\n',append=TRUE)
  cat(timestamp(),file=log,sep='\n',append=TRUE)
  for(l in unique(pannot$Analyte.Type)){
    
    pannot.sub <- pannot[(pannot$Analyte.Type) == l,]
    intersect.analyte <- intersect(dimnames(rawdata)[[2]],dimnames(pannot.sub)[[1]])
    
    processedData.list[[l]] <- processedData[,intersect.analyte]
    rawData.list[[l]]  <- rawdata[,intersect.analyte]
    
    if(! l %in% c("mRNA","protein"))
      stop(paste("No method for normalizing analyte type",l,"is available"))
    
    if(l == "mRNA"){
      
      #some default values set for the case where we don't normalize (check why we need to pass data into normalization func if don't want normalization)
      normmodule.arg.method <- "none"
      n.HKs <- 10
      auto.HKs <- TRUE
      codeset.HKs <- pannot[1,probe.annotation.id.column]
      
      if("mRNA" %in% normmodule.analytes.run){
        
        norm.opts <- tryCatch(expr = {get.mRNA.normalization.opt(raw.dat = as.matrix(rawData.list[[l]]),
                                                                 normmodule.norm.probes = normmodule.norm.probes,
                                                                 normmodule.nchoos = normmodule.nchoos,
                                                                 probe.annotation.id.column = probe.annotation.id.column,
                                                                 pannot = pannot)},
                              error = function(er) return(er),
                              finally ={})
        
        
        if(!class(norm.opts) == "try-error"){
          n.HKs <- norm.opts$n.HKs
          auto.HKs <- norm.opts$auto.HKs
          codeset.HKs <- norm.opts$codeset.HKs
          normmodule.arg.method <- norm.opts$HK.selection.method 
        }else{
          warnings.paragraph <- paste(warnings.paragraph,"\n",norm.opts)
        }
        
      }
      
      
      #Force exlusion of ERCCs from data to be normalized
      tmp <- as.matrix(rawData.list[[l]])
      ERCCs <- pannot[pannot$Control.Type %in% c("Negative","Positive"),probe.annotation.id.column]
      tmp.raw <- tmp[,!colnames(tmp) %in% ERCCs,drop=F]
      
      tmp <- processedData.list[[l]]
      tmp.processed <- tmp[,!colnames(tmp) %in% ERCCs,drop=F]
      normresult <- list()
      if(ncol(tmp.raw)>0 & ncol(tmp.processed)>0){
        
        normresult <- normalizerawdata(raw=tmp.raw,
                                       processed = tmp.processed,
                                       data.type = data.type,
                                       method=normmodule.arg.method,
                                       choose.method = "geNorm",
                                       n.HKs = n.HKs,
                                       auto.HKs = auto.HKs,
                                       codeset.HKs = codeset.HKs,
                                       path.to.normalization.results=path.to.normalization.results,
                                       log=log,
                                       plottypearg=plottypearg,
                                       path.results=path.results,
                                       path.inc=path.inc,
                                       prb.annots = pannot)
      }else{
        warnings.paragraph <- paste(warnings.paragraph,"Warning: No informative mRNA probes is left in this dataset","\n")
      }
    }
    if(l == "protein"){
      
      #Set the default to Processed data unless processed is different from raw and user explicitly asked
      #to use raw. The default is overwritten unless the user turn off the normalization
      normresult <- list()
      normresult$normalized.data <- log2(processedData.list[[l]]+1)
      if(!isTRUE(all.equal(rawData.list[[l]],processedData.list[[l]],check.attributes=F)) & data.type == "raw")
        normresult$normalized.data <- log2(rawData.list[[l]]+1)
      
      if("protein" %in% normmodule.analytes.run){
        
        # Manual option: User selected probes for normalization 
        #----------------------------------------------------------------------------------------------
        norm.ref.prot.names  <- get.probes.for.analyte (probes = normmodule.norm.probes,
                                                        probe.annotation.id.column = probe.annotation.id.column,
                                                        pannot = pannot,
                                                        analyte.type = "protein")
        
        neg.colnames <- as.character(pannot[which(pannot$Control.Type == "Protein_NEG"),probe.annotation.id.column])
        
        normresult <- preprocess.prot(raw.dat = rawData.list[[l]],
                                      neg.colnames = neg.colnames,
                                      norm.method = "geomean.stable1", 
                                      norm.ref.prot.names = norm.ref.prot.names,
                                      normmodule.nchoos = normmodule.nchoos,
                                      smp.annot = NULL,
                                      prb.annots = pannot,
                                      path.to.normalization.results = path.to.normalization.results,
                                      log = log,
                                      plottypearg = plottypearg,
                                      path.results = path.results,
                                      path.inc = path.inc)
      } else{
        warning("Warning: No protein normalization will be performed - assuming the input data is already normalized")
        cat("LOG: Warning: No protein normalization will be performed - assuming the input data is already normalized",file=log,sep='\n\n',append=TRUE)
        cat("document.write('<p>Warning: No protein normalization will be performed - assuming the input data is already normalized</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
        
      }
      
    }
    
    
    normData.list[[l]]      <- normresult$normalized.data
    normData.list.2[[l]]      <- normresult
    
  }
  
  # merge normalized data
  #----------------------
  cbind.all <- function(x, y){
    rowMatch <- match(rownames(x), rownames(y))
    cbind(x, y[rowMatch,])
  }
  
  e <- Reduce(cbind.all, normData.list)
  
  # Remove controls that are not being used
  #----------------------------------------
  control.probe.ids <- pannot[pannot$Is.Control == 1,probe.annotation.id.column]
  if(length(which(colnames(e)%in%control.probe.ids))>0)
    e <- e[,-which(colnames(e)%in%control.probe.ids)]
  
  # relabel e for report
  #---------------------
  e.relabeled <- e
  colnames(e.relabeled) <- paste(pannot[colnames(e),"Probe.Label"],pannot[colnames(e),"Analyte.Type"],sep="-")
  write.csv(e.relabeled, file = paste(path.to.normalization.results,"//ALL_normalized log2 data.csv",sep=""))
  
  
  #======================================================
  #            Overview Descriptive Analyses
  #======================================================
  QCresult <- NULL
  if(QCmodule.arg.run)
  {
    suppressWarnings(write.table(head(as.data.frame(annot)),file=log,sep='\t',append=TRUE))
    cat("running QC module",file=log,sep='\n',append=TRUE)
    #HTML NOTE - the HTML for the QC is embedded in the other R-file
    cat(timestamp(),file=log,sep='\n',append=TRUE)
    
    QCresult <- tryCatch(expr = {runQC(normalized=e, 
                                       rawdata=rawdata,
                                       thresholding.value=thresholding.value,
                                       probe.annot=pannot,
                                       rawDataAll.list=rawData.list,
                                       normalizedData.list=normData.list,
                                       HKnormalized=normData.list.2[["mRNA"]]$HKnormalized,
                                       HKs=normData.list.2[["mRNA"]]$HKs,
                                       covariates=as.data.frame(annot),
                                       covariates.type=sampleannot.variabletypes,
                                       prb.sets=prb.set.matrix.QC,
                                       annotcols=annotcols,
                                       annotcols2=annotcols2,
                                       codecols=codecols2,
                                       path.to.QC.results=path.to.QC.results,
                                       log=log,
                                       plottypearg=plottypearg,
                                       path.results=path.results,
                                       path.inc=path.inc,
                                       path.to.csvs=path.to.csvs)}
                         , error = function(er){
                           ertmp <- paste("Overview is aborted: the following error(s) occurred\n",er)
                           #cat(ertmp)
                           cat("document.write('<p> Overview aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                           class(ertmp) <- "error"
                           return(ertmp)
                         }
                         , finally = {})
    
    if(class(QCresult) == "error" )
    {
      print("Failed to run Overview")
      cat("LOG: Failed to run Overview",file=log,sep='\n',append=TRUE)
      cat(QCresult,file=log,sep='\n',append=TRUE)
      warnings.paragraph <- paste(warnings.paragraph,QCresult,"\n")
      QCmodule.arg.run <- FALSE
      QCresult <- NULL
    }
    
  }
  
  
  #======================================================
  #                DE Analysis                  
  #======================================================
  DEresults <- NULL
  DEmodule.arg.predictors = reconcile.variables.with.annot(annot,variables=DEmodule.arg.predictors,
                                                           variabletypes=NULL,
                                                           referencelevels=NULL)$variables
  
  if(length(DEmodule.arg.predictors)==0 & isTRUE(DEmodule.arg.run)){
    print("Failed to run DE Analysis")
    cat("LOG: Failed to run DE Analysis",file=log,sep='\n',append=TRUE)
    cat("document.write('<p>DE aborted: No valid predictor was selected for DE</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
    warnings.paragraph <- paste(warnings.paragraph,"No valid predictor was selected for DE. DE was aborted","\n")
    DEresults <- NULL
    DEmodule.arg.run=FALSE
  }
    
  
  if(DEmodule.arg.run)
  {
    #gene.set.matrix = pannot[,gene.set.columns]; dimnames(gene.set.matrix)[[1]] = dimnames(pannot)[[1]]
    
    cat("running DE module",file=log,sep='\n\n',append=TRUE)
    cat(timestamp(),file=log,sep='\n',append=TRUE)
    
    DEmodule.arg.confounders = reconcile.variables.with.annot(annot,variables=DEmodule.arg.confounders,
                                                              variabletypes=NULL,
                                                              referencelevels=NULL)$variables
    
    
    
    cat("predictors:",file=log,sep='\n',append=TRUE)
    suppressMessages(suppressWarnings(write.table(DEmodule.arg.predictors,file=log,sep='\t',append=TRUE)))
    cat("confounders:",file=log,sep='\n',append=TRUE)
    suppressMessages(suppressWarnings(write.table(DEmodule.arg.confounders,file=log,sep='\t',append=TRUE)))
    cat("predictor var types:",file=log,sep='\n',append=TRUE)
    suppressMessages(suppressWarnings(write.table(sampleannot.variabletypes[DEmodule.arg.predictors],file=log,sep='\t',append=TRUE)))
    cat("confounder var types:",file=log,sep='\n',append=TRUE)
    suppressMessages(suppressWarnings(write.table(sampleannot.variabletypes[DEmodule.arg.confounders],file=log,sep='\t',append=TRUE)))
    
    # Perform filtering out of low expressors and constant
    #-----------------------------------------------------
    filtered <- filter.out.low.counts(raw = rawdata,prb.annots=pannot,thresholding.frequency = thresholding.frequency, thresholding.value = thresholding.value,thresholding.perform = thresholding.perform)
    warnings.paragraph <- paste(warnings.paragraph,filtered$warnings,"\n")
    filtered.e <- e[,!filtered$prune[colnames(e)],drop=FALSE]
    const.prbs <- colnames(filtered.e)[apply(filtered.e,2,sd)==0]
    filtered.e <- filtered.e[,!colnames(filtered.e)%in%const.prbs]
    if(length(const.prbs)>0)
      warnings.paragraph <- paste(warnings.paragraph,paste(const.prbs,collapse = ","),"dropped from DE for being constant","\n")
    
    
    DEresults = tryCatch(expr = {run.DE.analyses(normalized=filtered.e,
                                                 predictors=DEmodule.arg.predictors,
                                                 confounders=DEmodule.arg.confounders,
                                                 annot=annot,
                                                 predictor.var.type=sampleannot.variabletypes[DEmodule.arg.predictors],
                                                 confounder.var.type=sampleannot.variabletypes[DEmodule.arg.confounders],
                                                 sampleannot.referencelevels=sampleannot.referencelevels,
                                                 pval.adjustment = DEmodule.arg.pval.adjustment,
                                                 prb.sets=prb.set.matrix[colnames(e[,!filtered$prune[colnames(e)],drop=FALSE]),,drop=FALSE],  
                                                 path.to.DE.results=path.to.DE.results,
                                                 log=log,
                                                 plottypearg=plottypearg,
                                                 path.results=path.results,
                                                 path.inc=path.inc,
                                                 path.to.csvs=path.to.csvs,
                                                 probe.annotation.id.column = probe.annotation.id.column,
                                                 prb.annots = pannot)},
                         error = function(er){
                           ertmp <- paste("DE is aborted: the following error(s) occurred in execution of DE\n",er)
                           #cat(ertmp)
                           cat("document.write('<p>DE aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                           class(ertmp) <- "DE.error"
                           return(ertmp)
                         },
                         finally = {})
      
 
    
    if(class(DEresults) == "DE.error" )
    {
      print("Failed to run DE Analysis")
      cat("LOG: Failed to run DE Analysis",file=log,sep='\n',append=TRUE)
      cat(DEresults,file=log,sep='\n',append=TRUE)
      warnings.paragraph <- paste(warnings.paragraph,DEresults,"\n")
      DEmodule.arg.run <- FALSE
      DEresults <- NULL
      
    }else{
      ## add gene sets corresponding to the DE hits:
      if(length(DEresults$sig.gene.sets)>0)
      {
        for(i in 1:length(DEresults$sig.gene.sets))
        {
          tempcolumn = 1*is.element(rownames(prb.set.matrix),DEresults$sig.gene.sets[[i]])
          prb.set.matrix.QC = cbind(prb.set.matrix.QC,tempcolumn); colnames(prb.set.matrix.QC)[ncol(prb.set.matrix.QC)]=names(DEresults$sig.gene.sets)[i]
        }
      }
      
    }
    
  }
  
  
  #======================================================
  #               Gene set Analysis
  #======================================================
  
  if(gsamodule.arg.run){
    wtmp <- NULL
    if(!DEmodule.arg.run){
      wtmp <- paste("Geneset analysis will be not be conducted because the required DE analysis was not performed\n")
      cat("document.write('<p>Geneset analysis will be not be conducted because the required DE analysis was not performed</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      gsamodule.arg.run <- FALSE
    }
    if(is.null(prb.set.matrix)){
      wtmp <- paste("Geneset analysis will be not be conducted because no geneset annotation is provided\n")
      cat("document.write('<p>Geneset analysis will be not be conducted because no geneset annotation is provided</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      gsamodule.arg.run <- FALSE
    }
    if(!is.null(wtmp)){
      warning(wtmp)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",wtmp,"\n")
    }
    rm(wtmp) 
  }
  
  
  if(gsamodule.arg.run)
  {
    cat("running gene set analysis module",file=log,sep='\n\n',append=TRUE)
    cat(timestamp(),file=log,sep='\n',append=TRUE)  
    
    
    gsaresults <- tryCatch(expr = run.gene.set.analyses(DEresults,prb.sets=prb.set.matrix[colnames(e[,!filtered$prune[colnames(e)],drop=FALSE]),,drop=FALSE],  #,
                                                        prb.set.names=colnames(prb.set.matrix),
                                                        pval.adjustment=DEmodule.arg.pval.adjustment,
                                                        log=log,plottypearg=plottypearg,path.results=path.results,path.inc=path.inc,
                                                        path.to.csvs=path.to.csvs,path.to.GSA.results=path.to.GSA.results,prb.annots=pannot)
                           , error=function(er){
                             ertmp <- paste("Geneset analysis is aborted: the following error(s) occurred\n",er)
                             #cat(ertmp)
                             cat("document.write('<p>Geneset Analysis is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                             class(ertmp) <- "error"
                             return(ertmp)
                           }
                           , finally = {})
    
    if(class(gsaresults) == "error" )
    {
      print("Failed to run Geneset Analysis")
      cat("LOG: Failed to run Geneset Analysis",file=log,sep='\n',append=TRUE)
      cat(gsaresults,file=log,sep='\n',append=TRUE)
      warnings.paragraph <- paste(warnings.paragraph,gsaresults,"\n")
      gsamodule.arg.run <- FALSE
      gsaresults <- NULL
    }
    
    
  }
  
  
  #======================================================
  #                 Pathview Plots
  #======================================================
  # each node represents a protein family; do the proteins in the assay need to be incorporated somehow?
  # each node can correspond to multiple genes; in which case it is an average of fold-changes or t-statistics
  # if incorporating the proteins, do the gene.names need to be corrected (official gene symbols)
  
  if(pathviewmodule.arg.run){
    
    if(is.null(KEGG.set.matrix)){
      pathviewmodule.arg.run=FALSE
      warning("Was unable to validate KEGG annoation - Pathview will not be run")
      cat("LOG: Warning: Was unable to validate KEGG annoation - Pathview will not be run",file=log,sep='\n',append=TRUE)
      cat("document.write('<p>Warning: Was unable to validate KEGG annoation - Pathview will not be run</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      paste(warnings.paragraph,"Warning:Was unable to validate KEGG annoation - Pathview will not be run","\n")
    }
    
    if(!DEmodule.arg.run){
      pathviewmodule.arg.run=FALSE
      warning("DE was not completed - Pathview will not be run")
      cat("LOG: Warning: DE was not completed - Pathview will not be run",file=log,sep='\n',append=TRUE)
      cat("document.write('<p>Warning: DE was not completed - Pathview will not be run</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      paste(warnings.paragraph,"Warning:DE was not completed - Pathview will not be run","\n")
    }
      
    
    if(length(pathviewmodule.arg.keggIDs)== 0){
      pathviewmodule.arg.run=FALSE
      warning("No pathway IDs were provided for pathview display")
      cat("LOG: Warning: no valid KEGG ID selected",file=log,sep='\n',append=TRUE)
      cat("document.write('<p>Warning: No valid KEGG ID is selected - Pathview will not be run</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      paste(warnings.paragraph,"Warning:No valid KEGG ID is selected - Pathview will not be run","\n")
    }
    
    if(!testCon()){
      pathviewmodule.arg.run=FALSE
      warning("Internet connection failure - Pathview will not be run")
      cat("LOG: Warning: Internet connection failure - Pathview will not be run",file=log,sep='\n',append=TRUE)
      cat("document.write('<p>Warning: Internet connection failure - Pathview will not be run</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      paste(warnings.paragraph,"arning: Internet connection failure - Pathview will not be run","\n")
    }
  }
  
  
  if(pathviewmodule.arg.run)
  {
    cat("running Pathview module",file=log,sep='\n\n',append=TRUE)
    cat(timestamp(),file=log,sep='\n',append=TRUE)
    # clean the custom IDs to make them parse right: rule: 5 digits, all numbers.
    #<----------------------------------
    
    pathview.results <- tryCatch(expr = run.pathview(DEresult=DEresults, 
                                                     keggIDs=pathviewmodule.arg.keggIDs,
                                                     colorscheme = pathviewmodule.arg.view.colorscheme,
                                                     pvalthresh=pathviewmodule.arg.pvalthres,
                                                     view=c("KEGG"),
                                                     predictors=DEmodule.arg.predictors,
                                                     path.to.pathview.results=path.to.pathview.results,
                                                     log=log,
                                                     path.results=path.results,
                                                     path.inc=path.inc,
                                                     path.to.csvs=path.to.csvs)
                                 , error = function(er){
                                   ertmp <- paste("Pathview Analysis is aborted: the following error(s) occurred\n",er)
                                   #cat(ertmp)
                                   cat("document.write('<p> Pathview Analysis is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                                   class(ertmp) <- "error"
                                   return(ertmp)
                                 }
                                 , finally = {})
    

    setwd(analysis.path)
    if(class(pathview.results) == "error"){
      print("Failed to run pathview")
      cat("LOG: Failed to run pathview",file=log,sep='\n',append=TRUE)
      cat(pathview.results,file=log,sep='\n',append=TRUE)
      pathviewmodule.arg.run <- FALSE
      pathview.results <- NULL
    }
    
    #note pathview result right not is just a warning
    if(length(pathview.results)>0){
      cat(pathview.results)
      warnings.paragraph <- paste(warnings.paragraph,pathview.results,"\n")
    }
    
  }
  
  
  #======================================================
  #                   Cell type Analysis
  #======================================================
  celltypescoreslist <- NULL
  if(celltypes.arg.run)  
  {
    tempcelltype = replace(pannot[,probe.annotation.celltype.id.column],pannot$Cell.Type=="",NA)
    names(tempcelltype) = dimnames(pannot)[[1]]
    
    celltypecontrasts = read.csv(celltypes.arg.contrastsfile,row.names=1)
    
    celltypes.arg.covariates = reconcile.variables.with.annot(annot,variables=celltypes.arg.covariates,
                                                              variabletypes=NULL,
                                                              referencelevels=NULL)$variables
    
    celltyperesult <- tryCatch(expr = immune.cell.abundance.analysis(data = e,
                                                                     prb.annot = pannot,
                                                                     celltype = tempcelltype,
                                                                     use.all.genes=(celltypes.arg.probeselection!="DynamicallySelect"),
                                                                     lowcor=.2,
                                                                     cell.type.pval.thresh=celltypes.arg.pvalthreshold,
                                                                     celltypes.arg.showraw=celltypes.arg.showraw,
                                                                     celltypes.arg.showrelative=celltypes.arg.showrelative,
                                                                     celltypecontrasts=celltypecontrasts,
                                                                     plot.against.covariates=length(celltypes.arg.covariates)>0,
                                                                     covariates=annot[,celltypes.arg.covariates,drop=FALSE],
                                                                     covariate.type=sampleannot.variabletypes[celltypes.arg.covariates],
                                                                     #annotcols=as.matrix(annotcols[,celltypes.arg.covariates]),
                                                                     annotcols=annotcols[,celltypes.arg.covariates,drop=FALSE],
                                                                     annotcols2=annotcols2[match(celltypes.arg.covariates,dimnames(annot)[[2]])],
                                                                     log=log,
                                                                     plottypearg=plottypearg,
                                                                     path.results=path.results,
                                                                     path.inc=path.inc,
                                                                     path.to.csvs=path.to.csvs,
                                                                     path.to.celltype.results=path.to.celltype.results)
                               , error=function(er){
                                 ertmp <- paste("cell type scoring is aborted: the following error(s) occurred\n",er)
                                 #cat(ertmp)
                                 cat("document.write('<p> Cell Type Analysis is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                                 class(ertmp) <- "error"
                                 return(ertmp)
                               }
                               , finally = {})
    
    if(class(celltyperesult) == "error"){
      print("Failed to run Cell Type analysis")
      cat("LOG: Failed to run Cell Type analysis",file=log,sep='\n',append=TRUE)
      cat(celltyperesult,file=log,sep='\n',append=TRUE)
      
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",celltyperesult,"\n")
      celltypes.arg.run <- FALSE
      celltyperesult <- NULL
    }else{
      celltypescoreslist = celltyperesult$celltypescoreslist
      warnings.paragraph <- paste(warnings.paragraph,celltyperesult$warnings.paragraph)
    }
    
  }
  
  
  #======================================================
  #                   Pathway Scoring Analysis
  #======================================================
  pathscore.results <- NULL
  if(pathscores.arg.run){
    wtmp <- character(0)
    if(is.null(prb.set.matrix)){
      wtmp0 <- paste("Pathway scoring can not be completed as there are no valid probesets defined")
      cat(paste("document.write('<p>",wtmp0,"</p>');"), file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      wtmp <- paste(wtmp,wtmp0,"\n")
      pathscores.arg.run <- FALSE
    }
    
    pathscores.arg.covariates <- pathscores.arg.covariates[pathscores.arg.covariates %in% colnames(annot)]
    if(length(pathscores.arg.covariates)==0){
      wtmp0 <- paste("Pathway scoring can not be completed as there are no valid covariates selected")
      cat(paste("document.write('<p>",wtmp0,"</p>');"), file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      wtmp <- paste(wtmp,wtmp0,"\n")
      pathscores.arg.run <- FALSE
    }

    
    if(!is.null(pathscores.arg.adjust.for)){
      if(any(!pathscores.arg.adjust.for %in% colnames(annot))){
        wtmp0 <- paste("Pathway scoring can not be completed due to invalid adjustment covaraite:",paste(pathscores.arg.adjust.for,collapse = ","))
        cat(paste("document.write('<p>,",wtmp0, ",</p>');"), file=paste(path.inc,"//status.js",sep=""),append=TRUE)
        wtmp <- paste(wtmp,wtmp0,"\n")
        pathscores.arg.run <- FALSE
      }
    }

    
    if(length(wtmp)>0){
      warning(wtmp)
      cat(wtmp,file=log,sep='\n',append=TRUE)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",wtmp,"\n")
    }
    rm(wtmp) 
  }
  
  if(pathscores.arg.run)  
  {
    # run the pathway scoring function:
    
    # data = normalized data matrix (no distinguishing between analyze types)
    # gene.sets.list: named list of gene set vectors for PC1 scoring
    # adjust.for: data frame of covariates to adjust for. ONLY used under PC1 <--------------------
    # signaturematrix: matrix of gene weights for predefined signatures
    # requireallgenes: T/F for whether to use incomplete gene signatures (ie those with some genes not in codeset)
    # mingenes: min number of genes for a geneset or signature to be calculated
    # prunedprobes: only removed from PC1 scoring
    
    
    if(!is.null(pathscores.arg.adjust.for))
      pathscores.arg.adjust.for <- annot[,pathscores.arg.adjust.for,drop=FALSE]
    
    pathscores.arg.covariates.type <- NA
    if(!is.null(pathscores.arg.covariates))
      pathscores.arg.covariates.type <- sampleannot.variabletypes[pathscores.arg.covariates]
    
    
    pathscore.results <- tryCatch(expr = run.pathway.scoring(data=e,
                                                            path.scoring.method=pathscores.arg.method,
                                                            #PC1 arguments:
                                                            prb.set.matrix=prb.set.matrix,
                                                            mingenes.pc1=5,
                                                            prunedprobes=NULL,
                                                            adjust.for=pathscores.arg.adjust.for, 
                                                            # existing scores arguments
                                                            signaturematrix.predefined=NULL,
                                                            requireallgenes=FALSE,
                                                            mingenes.existing=5, 
                                                            # other arguments:
                                                            plot.against.covariates=length(pathscores.arg.covariates)>0,
                                                            covariates=as.data.frame(annot[,pathscores.arg.covariates,drop=F]),
                                                            covariate.type=pathscores.arg.covariates.type,
                                                            annotcols=annotcols,
                                                            annotcols2=annotcols2,
                                                            log=log,
                                                            plottypearg=plottypearg,
                                                            path.results=path.results,
                                                            path.inc=path.inc,
                                                            path.to.csvs=path.to.csvs,
                                                            path.to.pathscore.results=path.to.pathscore.results)
                                  ,error = function(er){
                                    ertmp <- paste("Pahtway scoring is aborted: the following error(s) occurred\n",er)
                                    #cat(ertmp)
                                    cat("document.write('<p> Pathway Scoring is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                                    class(ertmp) <- "error"
                                  }
                                  ,finally ={} )
    
    if(class(pathscore.results) == "error"){
      print("Failed to run Pathway Scoring Analysis")
      cat("LOG: Failed to run Pahtway Scoring Analysis",file=log,sep='\n',append=TRUE)
      cat("document.write('<p> Failed to run Pahtway Scoring Analysis due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
      cat(pathscore.results,file=log,sep='\n',append=TRUE)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",pathscore.results,"\n")
      pathscore.results <- NULL
      pathscores.arg.run <- FALSE
    }
    
  }
  
  
  #======================================================
  #                   SPD Analysis
  #======================================================
  SPD.dir <- NULL
  setwd(analysis.path)
  
  if(SPDarg.select.probe.analysis.run){
  
    wtmp <- validate.SPD.arguments(dat = e,smp.annots = annot)
    if(length(wtmp)>0)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",wtmp)
    
    
    if(length(SPDarg.probe.names.to.describe)>0){
      SPD.dir <- tryCatch(expr = {run.SPD.module(dat = e,
                                                 prb.annots = pannot,
                                                 smp.annots = annot,
                                                 color.cat.annots = annotcols2,
                                                 probe.ids.to.describe = SPDarg.probe.names.to.describe,
                                                 describe.probes.by = SPDarg.describe.probes.by,
                                                 stratify.probe.desription.by = NULL,
                                                 trend.by = SPDarg.trend.expression.by,
                                                 order.trend.by = SPDarg.order.expression.trend.by,
                                                 stratify.trend.by = SPDarg.stratify.expression.trend.by,
                                                 generate.interactnet.plot= SPDarg.interactnet.run,
                                                 interactnet.adjustfor = SPDarg.interactnet.adjustfor,
                                                 file.extension = plottypearg,
                                                 path.to.SPD.results = path.to.SPD.results)}
                          , error = function(er){
                            ertmp <- paste("Probe Descriptive Analysis is aborted: the following error(s) occurred\n",er)
                            #cat(ertmp)
                            cat("document.write('<p> Probe Descriptive Analysis is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                            class(ertmp) <- "SPD.error"
                            return(ertmp)
                          }
                          , finally ={})
      # Generate error message if error was caught
      #-------------------------------------------
      if(class(SPD.dir) == "SPD.error"){
        print("Failed in Probe Descriptive Analysis")
        cat("LOG: Failed in Probe Descriptive Analysis",file=log,sep='\n',append=TRUE)
        cat(SPD.dir,file=log,sep='\n',append=TRUE)
        warnings.paragraph <- paste(warnings.paragraph,"Warning:",SPD.dir,"\n")
        SPD.dir <- NULL
        # SPDarg.select.probe.analysis.run <- FALSE
      }
      
    }else{
      wtmp <- "No probes remain in the set of probes to describe. Therefore, SPD analysis will not be completed"
      warning(wtmp)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",wtmp,"\n")
    }
    
    if(is.null(SPD.dir))
      SPDarg.select.probe.analysis.run <- FALSE
    
  }# End of SPD
  
  
  #======================================================
  #                   RPD Analysis
  #======================================================
  setwd(analysis.path)
  
  RPD.dir <- NULL
  
  if(RPDarg.run){
    wtmp <- validate.RPD.arguments(dat = e,prb.annots = pannot,smp.annots = annot)
    if(length(wtmp)>0)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",wtmp)
    
    
    
    RPD.dir <- tryCatch(expr = {run.RPD(dat = e, 
                                        prb.annots = pannot,
                                        smp.annots = annot,
                                        prb.grp.members = RPDarg.probe.pairs.to.describe,
                                        var.to.describe.probes.by = RPDarg.describe.probes.by,
                                        var.to.trend.by = RPDarg.trend.expression.by,
                                        var.to.order.obs.by = RPDarg.order.expression.trend.by,
                                        var.to.stratify.trend.by = RPDarg.stratify.expression.trend.by,
                                        color.cat.annots = annotcols2,
                                        file.extension = plottypearg,
                                        path.to.RPD.results = path.to.RPD.results)}
                        , error = function(er){
                          ertmp <- paste("Related Analytes Analysis is aborted: the following error(s) occurred\n",er)
                          #cat(ertmp)
                          cat("document.write('<p> Related Analytes Analysis is aborted due to error</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
                          class(ertmp) <- "SPD.error"
                          return(ertmp)
                        }
                        , finally ={})
    # Generate error message if error was caught
    #-------------------------------------------
    if(class(RPD.dir) == "SPD.error"){
      print("Failed in Related Analytes Analysis")
      cat("LOG: Failed in Related Analytes Analysis",file=log,sep='\n',append=TRUE)
      cat(RPD.dir,file=log,sep='\n',append=TRUE)
      warnings.paragraph <- paste(warnings.paragraph,"Warning:",RPD.dir,"\n")
      RPD.dir <- NULL
    }
    
    if(is.null(RPD.dir))
      RPDarg.run <- FALSE
    
  }# End of RPD
  
  
  #======================================================
  #       Copy results into user-specified direcotry
  #======================================================
  cat("making results file directories",file=log,sep='\n\n',append=TRUE)
  
  # append the date and time to the analysis name
  # for the purpose of copying files:
  #----------------------------------------------
  advanced.analysis.name.for.file.copy <- paste(advanced.analysis.name,gsub(":","-",substr(Sys.time(),1,16)))
  results.basedir <- paste(path.results.copy,"//",advanced.analysis.name.for.file.copy,sep="")
  
  
  
  # make directories:
  #------------------
  dir.create(results.basedir)
  results.copydir <- paste(results.basedir, "//", "results", sep="")
  
  dir.create(results.copydir)
  
  # and copy results:
  #------------------
  cat("copying results files",file=log,sep='\n\n',append=TRUE)
  setwd(path.results)
  tfiles = dir()[-grep(pattern = "log|csvs",dir())]
  for(j in 1:length(tfiles))
  {
    invisible(file.copy(from = tfiles[j],to = results.copydir ,recursive = T))
  }
  
  # copy the resources
  results.copydir = paste(results.basedir, "//", "resources", sep="")
  dir.create(results.copydir)
  
  setwd(path.resources)
  tfiles = dir()
  for(j in 1:length(tfiles))
  {
    invisible(file.copy(from = tfiles[j],to = results.copydir ,recursive = T))
  }
  
  setwd(analysis.path)
  
  
  
  #======================================================
  #     Bundle the output of analysis into a list
  #======================================================
  
  #set some default values
  if(!exists("normresult"))
    normresult <- NULL
  if(!exists("annot.obj"))
    annot.obj <- NULL
  if(!exists("QCresult"))
    QCresult <- NULL
  if(!exists("DEresults"))
    DEresults<- NULL
  if(!exists("gsaresults"))
    gsaresults<- NULL
  if(!exists("pathview.results"))
    pathview.results<- NULL
  if(!exists("celltyperesult"))
    celltyperesult<- NULL
  if(!exists("celltypescoreslist"))
    celltypescoreslist<- NULL
  if(!exists("pathscore.results"))
    pathscore.results <- NULL
  
  # return all analysis results:
  analysisresults = list(annot.obj =  annot.obj,
                         normresult = normData.list.2[[1]],   # mRNA only
                         e = e,
                         QCresult = QCresult,
                         DEresults = DEresults,
                         gsaresults = gsaresults,
                         pathview.results = pathview.results,
                         celltyperesult = celltyperesult,
                         celltypescoreslist = celltypescoreslist,
                         QCmodule.arg.run = QCmodule.arg.run,
                         normmodule.arg.run=normmodule.arg.run,
                         DEmodule.arg.run=DEmodule.arg.run,
                         DEmodule.arg.predictors=DEmodule.arg.predictors,
                         DEmodule.arg.confounders=DEmodule.arg.confounders,
                         gsamodule.arg.run=gsamodule.arg.run,
                         pathviewmodule.arg.run=pathviewmodule.arg.run,
                         pathviewmodule.arg.keggnames=pathviewmodule.arg.keggNames,#pathviewmodule.arg.keggnames,
                         keggIDs=pathviewmodule.arg.keggIDs,
                         celltypes.arg.run = celltypes.arg.run,
                         pathscores.arg.run = pathscores.arg.run,
                         SPD.dir=SPD.dir,
                         SPDarg.select.probe.analysis.run = SPDarg.select.probe.analysis.run,
                         RPD.dir=RPD.dir,
                         RPDarg.run = RPDarg.run,
                         warnings.paragraph=warnings.paragraph,
                         celltypes.arg.covariates=celltypes.arg.covariates,
                         probe.set.matrix.QC=prb.set.matrix.QC,
                         arg.env = arg.env,
                         prb.annots = pannot,
                         results.basedir = results.basedir)
  
  return(analysisresults)
}




#======================================================
#           Analysis Initialization
#======================================================
analysis.path <- getwd()
path.results <-  path.results0 <- paste(getwd(),"/results",sep="");dir.create(path.results)

log <- paste(analysis.path,"//log_file.txt",sep="")

# Create directory paths
#-----------------------
path.rsource = paste(getwd(), "//R_source", sep="");dir.create(path.rsource)
path.resources = paste(getwd(),"//resources",sep="");dir.create(path.resources)
path.js = paste(path.resources,"//js",sep="");dir.create(path.js)
path.css = paste(path.resources,"//css",sep="");dir.create(path.css)
path.img = paste(path.resources,"//img",sep="");dir.create(path.img)
path.inc = paste(path.resources,"//inc",sep="");dir.create(path.inc)


path.to.normalization.results = paste(path.results,"//Normalization",sep=""); dir.create(path.to.normalization.results)
path.to.csvs = paste(path.results,"//csvs",sep=""); dir.create(path.to.csvs)
path.to.DE.results = paste(path.results,"//DE",sep=""); dir.create(path.to.DE.results)
path.to.SPD.results = paste(path.results,"//SPD",sep=""); dir.create(path.to.SPD.results)
path.to.RPD.results = paste(path.results,"//RPD",sep=""); dir.create(path.to.RPD.results)
path.to.QC.results = paste(path.results,"//QC",sep=""); dir.create(path.to.QC.results)
path.to.pathview.results = paste(path.results,"//Pathview",sep=""); dir.create(path.to.pathview.results)
path.to.GSA.results = paste(path.results,"//Gene set analysis",sep=""); dir.create(path.to.GSA.results)
path.to.celltype.results = paste(path.results,"//cell types",sep=""); dir.create(path.to.celltype.results)
path.to.pathscore.results = paste(path.results,"//pathway scoring",sep=""); dir.create(path.to.pathscore.results)


# Read in the argument file
#--------------------------
cat("loading arguments file",file=log,sep='\n')
arg.env <- source.arguments() # NOTE: arg.env is for convenience it is an environment with all the values of argument files and those modified
plottypearg = unique(c("png",plottypearg))


# Copy the other R files into their folder
#-----------------------------------------
flistr <- list.files(paste(path.pipeline.base,"//R_source",sep=""),"^.+[.][rR]$",full.names=TRUE)
invisible(file.copy(flistr,path.rsource))


# Load the source files
#----------------------
source("R_source/source - utils.r")
source("R_source/source - DE.R")
source("R_source/source - normalization.r") 
source("R_source/source - SPD.R")
source("R_source/source - RPD.R")
source("R_source/source - load_n_clean.R")     # <- add it to util?
source("R_source/species_specific.r")
source("R_source/source - pathview.r")
source("R_source/source - GSA.R")
source("R_source/source - QC.r")
source("R_source/source - signatures.r")
source("R_source/source - cell type profiling.R")
# source("R_source/source - IO software specific.r")
source("R_source/advanced_analysis_report.r")
source("R_source/advanced_analysis_report_extended2.r")
source("R_source/write_universal_analysis_report.R")

cat("loaded source",file=log,sep='\n\n',append=TRUE)


#copy static files
#-----------------
flistcss <- list.files(paste(path.pipeline.base,"//css",sep=""),"^.+[.]css$",full.names=TRUE)
invisible(file.copy(flistcss,path.css))
flistjs <- list.files(paste(path.pipeline.base,"//js",sep=""),"^.+[.]js$",full.names=TRUE)
invisible(file.copy(flistjs,path.js))
flistimg <- list.files(paste(path.pipeline.base,"//img",sep=""),"^.+[.].+$",full.names=TRUE)
invisible(file.copy(flistimg,path.img))

#======================================================
#                   Run the analyses
#======================================================

result <- tryCatch(expr = run.analysis(),
                   error = function(er){
                     ertmp <- paste("Analysis aborted entirely due to critical failures (see the logs)\n",er)
                     #cat(ertmp)
                     class(ertmp) <- "error"
                     return(ertmp)
                   },
                   finally = {})
warnings()
if(class(result) == "error"){
  print("Analysis could not be completed")
  cat(result)
  cat(paste("LOG: Analysis could not be completed",result),file=log,sep='\n',append=TRUE)
  cat(result,file=log,sep='\n',append=TRUE)
  cat("document.write('<p>Analysis could not be completed</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
  tmp <- unlist(strsplit(result,split = "\\n"))
  tmp <- tmp[tmp!="" & tmp!=" "]
  for(err.i in tmp){
    cat(paste("document.write('<p>",err.i,"</p>');"), file=paste(path.inc,"//status.js",sep=""),append=TRUE)
  }
  
  
}else{
  #======================================================
  #                 Generate HTML report
  #======================================================
  # initialize the reporting api
  #------------------------------
  # This function call adds the objects 
  # 1- advAnalysisReport 
  # 2- paramReport
  # to the environment from which it was called from (here: globalenv)
  
  initAdvAnalysisReport(path.pipeline.base) 
  
  reportresult = try(write.univesal.analysis.report(result = result,advAnalysisReport = advAnalysisReport, prb.annots = result$prb.annots, arg.env=result$arg.env))
  invisible(file.copy(from = "AdvAnalysisReport.html", to = result$results.basedir))
  
  warnings()
  if(class(reportresult) == "try-error"){
    print("Failed to write analysis report")
    cat("LOG: Failed to write analysis report",file=log,sep='\n',append=TRUE)
    cat(reportresult,file=log,sep='\n',append=TRUE)
    cat("document.write('<p>Analysis could not be completed</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
  }else {
    print("Analysis is complete")
    cat("LOG:Analysis is complete",file=log,sep='\n',append=TRUE)
    cat("document.write('<p>Analysis is complete</p>');", file=paste(path.inc,"//status.js",sep=""),append=TRUE)
    cat("window.location = \"AdvAnalysisReport.html\";", file=paste(path.inc,"//redirect.js",sep=""),append=TRUE)  
  }
  
}
setwd(analysis.path)



